const express = require("express");
const morgan = require("morgan");
const app = express();
const cors = require("cors");
const path = require("path");
const globalErrorHandler = require("./controllers/errorController");
const AppError = require("./utils/appError");

// --- Route Imports ---
const userRouter = require("./routes/userRoutes");
const courseRouter = require("./routes/courseRoutes");
const campusRouter = require("./routes/campusRoutes");
const staffRouter = require("./routes/staffRoutes");
const staffRoleRouter = require("./routes/staffRoleRoutes");
const instructorRouter = require("./routes/instructorRoutes");
const categoryRouter = require("./routes/categoryRoutes");
const partnerRouter = require("./routes/partnerRoutes");
const cartRouter = require("./routes/cartRoutes");
const wishlistRouter = require("./routes/wishlistRoutes");
const couponRouter = require("./routes/couponRoutes");
const orderRouter = require("./routes/orderRoutes");
const faqRouter = require("./routes/faqRoutes");
const blogRouter = require("./routes/blogRoutes");
const contactRoutes = require("./routes/contactRoutes");
const paymentRouter = require("./routes/paymentRoutes");
const eventRouter = require("./routes/eventRoutes");
const subscriberRouter = require("./routes/subscriberRoutes");
const infoRouter = require("./routes/infoRoutes");
const googleReviewRouter = require("./routes/googleReviewRoutes");
// --- Middleware ---
console.log(`Environment: ${process.env.NODE_ENV}`);
if (process.env.NODE_ENV === "development") {
  app.use(morgan("dev"));
}

// 🔥 UPDATED CORS: Added your IP Address
app.use(
  cors({
    origin: [
      "https://impactintl.edu.sa/",
      "https://dashboard.impactintl.edu.sa/",
    ],
    credentials: true,
    methods: ["GET", "POST", "PUT", "PATCH", "DELETE", "OPTIONS"],
    allowedHeaders: ["Content-Type", "Authorization", "x-api-key"],
  }),
);

// ✅ Limits for Body Parser
app.use(express.json({ limit: "50mb" }));
app.use(express.urlencoded({ limit: "50mb", extended: true }));

// ✅ Limits for Body Parser
app.use(express.json({ limit: "50mb" }));
app.use(express.urlencoded({ limit: "50mb", extended: true }));

// --- Static Files ---

// 1. User Images
app.use(
  "/img/users",
  express.static(path.join(__dirname, "storage/storageMedia/users")),
);

// 2. Course Media
app.use(
  "/img/courses",
  express.static(path.join(__dirname, "storage/storageMedia/courses/images")),
);
app.use(
  "/video/courses",
  express.static(path.join(__dirname, "storage/storageMedia/courses/videos")),
);

// 3. Campus Media (Images & Videos)
app.use(
  "/img/campus",
  express.static(path.join(__dirname, "storage/storageMedia/campus/images")),
);
app.use(
  "/video/campus",
  express.static(path.join(__dirname, "storage/storageMedia/campus/videos")),
);

// 4. Staff Media
app.use(
  "/img/staff",
  express.static(path.join(__dirname, "storage/storageMedia/staff")),
);

// 5. Category Media
app.use(
  "/img/categories",
  express.static(path.join(__dirname, "storage/storageMedia/categories")),
);

// 6. Partner Media
app.use(
  "/img/partners",
  express.static(path.join(__dirname, "storage/storageMedia/partners")),
);
// 7. Blog Media
app.use(
  "/img/blogs",
  express.static(path.join(__dirname, "storage/storageMedia/blogs")),
);

// --- API Routes ---
app.use("/api/users", userRouter);
app.use("/api/courses", courseRouter);
app.use("/api/campus", campusRouter);
app.use("/api/staff", staffRouter);
app.use("/api/roles", staffRoleRouter);
app.use("/api/instructors", instructorRouter);
app.use("/api/categories", categoryRouter);
app.use("/api/partners", partnerRouter);
app.use("/api/cart", cartRouter);
app.use("/api/wishlist", wishlistRouter);
app.use("/api/coupons", couponRouter);
app.use("/api/orders", orderRouter);
app.use("/api/faqs", faqRouter);
app.use("/api/blogs", blogRouter);
app.use("/api/contact", contactRoutes);
app.use("/api/payment", paymentRouter);
app.use("/api/events", eventRouter);
app.use("/api/subscribers", subscriberRouter);
app.use("/api/info", infoRouter);
app.use("/api/reviews", googleReviewRouter);
//Handle Unhandled Routes (404)
app.all(/(.*)/, (req, res, next) => {
  next(new AppError(`Can't find ${req.originalUrl} on this server!`, 404));
});

// GLOBAL ERROR HANDLER (This makes the AppError work!)
app.use(globalErrorHandler);

module.exports = app;
